import axios from 'axios';

const api = axios.create({
  baseURL: '/api',
  headers: {
    'Accept': 'application/json',
    'Content-Type': 'application/json',
  },
});

// Add auth token automatically
api.interceptors.request.use(config => {
  const token = localStorage.getItem('auth_token');
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});

// Handle 401 globally (redirect to login)
api.interceptors.response.use(
  response => response,
  error => {
    if (error.response?.status === 401) {
      localStorage.removeItem('auth_token');
      localStorage.removeItem('user');
      window.location.href = '/login';
    }
    return Promise.reject(error);
  }
);

// Auth
export const auth = {
  login: (email, password) => api.post('/login', { email, password }),
  register: (name, email, password) => api.post('/register', { name, email, password }),
  logout: () => api.post('/logout'),
  user: () => api.get('/user'),
};

// Super Admin (tenants)
export const tenants = {
  list: () => api.get('/tenants'),
  create: (name, domain, adminName, adminEmail, adminPassword) => 
    api.post('/tenants', { name, domain, adminName, adminEmail, adminPassword }),
};

// Materials
export const materials = {
  profiles: {
    list: () => api.get('/profiles'),
    create: (name, pricePerM) => api.post('/profiles', { name, pricePerM }),
    update: (id, name, pricePerM) => api.put(`/profiles/${id}`, { name, pricePerM }),
    delete: (id) => api.delete(`/profiles/${id}`),
  },
  irons: {
    list: () => api.get('/irons'),
    create: (name, pricePerM) => api.post('/irons', { name, pricePerM }),
    update: (id, name, pricePerM) => api.put(`/irons/${id}`, { name, pricePerM }),
    delete: (id) => api.delete(`/irons/${id}`),
  },
  glassTypes: {
    list: () => api.get('/glass-types'),
    create: (name, pricePerM2) => api.post('/glass-types', { name, pricePerM2 }),
    update: (id, name, pricePerM2) => api.put(`/glass-types/${id}`, { name, pricePerM2 }),
    delete: (id) => api.delete(`/glass-types/${id}`),
  },
  accessories: {
    list: () => api.get('/accessories'),
    create: (name, unit, price) => api.post('/accessories', { name, unit, price }),
    update: (id, name, unit, price) => api.put(`/accessories/${id}`, { name, unit, price }),
    delete: (id) => api.delete(`/accessories/${id}`),
  },
};

// Offers
export const offers = {
  list: () => api.get('/offers'),
  create: (offer) => api.post('/offers', offer),
  show: (id) => api.get(`/offers/${id}`),
  update: (id, offer) => api.put(`/offers/${id}`, offer),
  delete: (id) => api.delete(`/offers/${id}`),
};

// Positions (nested under offers)
export const positions = {
  create: (offerId, position) => api.post(`/offers/${offerId}/positions`, position),
  update: (offerId, positionId, position) => api.put(`/offers/${offerId}/positions/${positionId}`, position),
  delete: (offerId, positionId) => api.delete(`/offers/${offerId}/positions/${positionId}`),
};

export default api;