import React, { useState } from "react";
import { UI } from "../../constants/ui";
import { Btn } from "../../components/ui/Btn";
import { Pill } from "../../components/ui/Pill";
import { Modal } from "../../components/ui/Modal";
import { money } from "../../utils/money";

import { OfferOverview } from "./OfferOverview";
import { PVCEditor } from "../editor/PVCEditor";

export function OfferWorkspace({ offer, materials, onBack, onOpenMaterials, onUpdateOffer, onDeleteOffer }) {
  const [tab, setTab] = useState("overview");
  const [dangerOpen, setDangerOpen] = useState(false);

  if (!offer) {
    return (
      <div style={{ padding: 24 }}>
        <Btn onClick={onBack}>← Kthehu</Btn>
        <div style={{ marginTop: 10, color: UI.muted }}>Oferta nuk u gjet.</div>
      </div>
    );
  }

  const positions = offer.positions || [];
  const offerTotal = positions.reduce((s, p) => s + (p.total || 0), 0);

  const profileName = materials.profiles.find((p) => p.id === offer.profileId)?.name || "—";
  const ironName = materials.irons.find((i) => i.id === offer.ironId)?.name || "—";

  return (
    <div style={{ minHeight: "100vh", background: UI.bg }}>
      <div className="stickyHeader" style={{ zIndex: 25 }}>
        <div style={{ padding: "12px 14px", display: "flex", alignItems: "center", justifyContent: "space-between", gap: 12, flexWrap: "wrap" }}>
          <div style={{ display: "flex", alignItems: "center", gap: 10, minWidth: 0, flexWrap: "wrap" }}>
            <Btn variant="ghost" onClick={onBack} style={{ padding: "10px 12px" }}>
              ← Dashboard
            </Btn>
            <Btn variant="soft" onClick={onOpenMaterials} style={{ padding: "10px 12px" }}>
              ⚙ Materialet
            </Btn>

            <div style={{ minWidth: 0 }}>
              <div style={{ display: "flex", alignItems: "center", gap: 10, flexWrap: "wrap" }}>
                <div style={{ fontWeight: 950, fontSize: 14, whiteSpace: "nowrap", overflow: "hidden", textOverflow: "ellipsis" }}>
                  {offer.name}
                </div>
                <Pill tone="soft">Pozicione: {positions.length}</Pill>
                <Pill tone="green">€ {money(offerTotal)}</Pill>
              </div>
              <div style={{ fontSize: 12, color: UI.muted, marginTop: 2 }}>
                Profil: {profileName} • Armaturë: {ironName} • Ngjyra: {offer.color}
              </div>
            </div>
          </div>

          <div style={{ display: "flex", gap: 8, alignItems: "center", flexWrap: "wrap" }}>
            <div style={{ display: "flex", background: UI.soft, border: `1px solid ${UI.border}`, borderRadius: 14, padding: 4 }}>
              <button
                onClick={() => setTab("overview")}
                style={{
                  border: "none",
                  background: tab === "overview" ? "#fff" : "transparent",
                  borderRadius: 10,
                  padding: "8px 10px",
                  fontWeight: 900,
                  fontSize: 12,
                  cursor: "pointer",
                  border: tab === "overview" ? `1px solid ${UI.border}` : "1px solid transparent",
                }}
              >
                Përmbledhje
              </button>
              <button
                onClick={() => setTab("editor")}
                style={{
                  border: "none",
                  background: tab === "editor" ? "#fff" : "transparent",
                  borderRadius: 10,
                  padding: "8px 10px",
                  fontWeight: 900,
                  fontSize: 12,
                  cursor: "pointer",
                  border: tab === "editor" ? `1px solid ${UI.border}` : "1px solid transparent",
                }}
              >
                Editori
              </button>
            </div>

            <Btn variant="danger" onClick={() => setDangerOpen(true)}>
              Fshij ofertën
            </Btn>
          </div>
        </div>
      </div>

      {tab === "overview" ? (
        <OfferOverview offer={offer} materials={materials} onUpdateOffer={onUpdateOffer} onOpenEditor={() => setTab("editor")} />
      ) : (
        <PVCEditor offer={offer} materials={materials} onUpdateOffer={onUpdateOffer} onBackToOverview={() => setTab("overview")} />
      )}

      <Modal
        open={dangerOpen}
        title="Fshij ofertën"
        onClose={() => setDangerOpen(false)}
        footer={
          <div style={{ display: "flex", justifyContent: "space-between", gap: 10 }}>
            <Btn onClick={() => setDangerOpen(false)}>Anulo</Btn>
            <Btn
              variant="danger"
              onClick={() => {
                setDangerOpen(false);
                onDeleteOffer?.();
              }}
            >
              Fshij përgjithmonë
            </Btn>
          </div>
        }
      >
        <div style={{ fontSize: 13, color: UI.ink, lineHeight: 1.5 }}>
          Kjo do ta fshijë përfundimisht <b>{offer.name}</b> dhe të gjitha pozicionet brenda saj.
        </div>
      </Modal>
    </div>
  );
}
