import React, { useEffect, useMemo, useState } from "react";
import Modal from "../components/common/Modal";
import { Btn, Pill, Select, Field } from "../components/common/UI";
import { UI } from "../constants/pvc";
import { money } from "../utils/helpers";

export default function OffersDashboard({
  offers,
  materials,
  onOpenOffer,
  onCreateOffer,
  onOpenMaterials,
  logoSrc,
  userLabel = "User",
}) {
  const [q, setQ] = useState("");
  const [sort, setSort] = useState("updated_desc");
  const [createOpen, setCreateOpen] = useState(false);

  // ✅ Create Offer Draft (native inputs -> no focus issue)
  const [name, setName] = useState("");
  const [description, setDescription] = useState("");
  const [margin, setMargin] = useState(20);
  const [color, setColor] = useState("[BB] Bardhë-Bardhë");
  const [profileId, setProfileId] = useState(materials.profiles[0]?.id || "");
  const [ironId, setIronId] = useState(materials.irons[0]?.id || "");

  useEffect(() => {
    setProfileId((p) => p || materials.profiles[0]?.id || "");
    setIronId((i) => i || materials.irons[0]?.id || "");
  }, [materials.profiles, materials.irons]);

  const counts = useMemo(() => {
    return {
      prof: materials.profiles.length,
      iron: materials.irons.length,
      glass: materials.glassTypes.length,
      acc: materials.accessories.length,
    };
  }, [materials]);

  // ✅ IMPORTANT: total includes qty
  const totalsByOfferId = useMemo(() => {
    const map = new Map();
    for (const o of offers) {
      const sum = (o.positions || []).reduce((s, p) => {
        const qty = Math.max(1, Number(p.qty || 1));
        return s + Number(p.total || 0) * qty;
      }, 0);
      map.set(o.id, sum);
    }
    return map;
  }, [offers]);

  const totalAll = useMemo(() => {
    let t = 0;
    for (const o of offers) t += totalsByOfferId.get(o.id) || 0;
    return t;
  }, [offers, totalsByOfferId]);

  const totalPositions = useMemo(() => offers.reduce((s, o) => s + (o.positions || []).length, 0), [offers]);

  const avgOffer = useMemo(() => {
    const c = offers.length || 1;
    return totalAll / c;
  }, [offers.length, totalAll]);

  const filtered = useMemo(() => {
    const s = q.trim().toLowerCase();
    if (!s) return offers;
    return offers.filter((o) => {
      const n = (o.name || "").toLowerCase();
      const d = (o.description || "").toLowerCase();
      return n.includes(s) || d.includes(s);
    });
  }, [offers, q]);

  const sorted = useMemo(() => {
    const arr = [...filtered];
    const getUpdated = (o) => (o.updatedAt ? new Date(o.updatedAt).getTime() : 0);
    const getPrice = (o) => totalsByOfferId.get(o.id) || 0;
    const getName = (o) => (o.name || "").toLowerCase();

    arr.sort((a, b) => {
      if (sort === "name_asc") return getName(a).localeCompare(getName(b));
      if (sort === "name_desc") return getName(b).localeCompare(getName(a));
      if (sort === "updated_asc") return getUpdated(a) - getUpdated(b);
      if (sort === "updated_desc") return getUpdated(b) - getUpdated(a);
      if (sort === "price_asc") return getPrice(a) - getPrice(b);
      if (sort === "price_desc") return getPrice(b) - getPrice(a);
      return 0;
    });

    return arr;
  }, [filtered, sort, totalsByOfferId]);

  const openCreate = () => {
    setName("");
    setDescription("");
    setMargin(20);
    setColor("[BB] Bardhë-Bardhë");
    setProfileId(materials.profiles[0]?.id || "");
    setIronId(materials.irons[0]?.id || "");
    setCreateOpen(true);
  };

  const createOffer = () => {
    const n = (name || "").trim();
    if (!n) return;

    onCreateOffer({
      name: n,
      description: (description || "").trim(),
      margin: Number(margin) || 0,
      color,
      profileId,
      ironId,
      updatedAt: new Date().toISOString(),
    });

    setCreateOpen(false);
  };

  return (
    <div className="od2Root">
      <style>{styles(UI)}</style>

      {/* Navbar */}
      <header className="od2Nav">
        <div className="od2NavInner">
          <div className="od2NavLeft">
            <div className="od2LogoWrap" title="Logo">
              {logoSrc ? <img src={logoSrc} alt="Logo" className="od2LogoImg" style={{width: "100px"}} /> : <div className="od2LogoFallback">PVC</div>}
            </div>

            <nav className="od2Links" aria-label="Navigation">
              <button className="od2Link od2Active" type="button">
                <span className="od2LinkIcon">
                  <IconHome />
                </span>
                Dashboard
              </button>

              <button
                className="od2Link"
                type="button"
                onClick={onOpenMaterials}
                title={`P:${counts.prof} • A:${counts.iron} • X:${counts.glass} • AK:${counts.acc}`}
              >
                <span className="od2LinkIcon">
                  <IconSliders />
                </span>
                Materialet
              </button>
            </nav>
          </div>

          <div className="od2NavRight">
            <button className="od2UserBtn" type="button" title={userLabel} aria-label="User">
              <span className="od2UserIcon">
                <IconUser />
              </span>
            </button>
          </div>
        </div>
      </header>

      {/* Main */}
      <main className="od2Main">
        <div className="od2Full">
          {/* Stats */}
          <section className="od2Stats">
            <StatCard label="Totali" value={`€ ${money(totalAll)}`} hint="Shuma e pozicioneve (me sasi)" icon={<IconEuro />} tone="blue" />
            <StatCard label="Oferta" value={`${offers.length}`} hint={`Të filtruara: ${sorted.length}`} icon={<IconFile />} tone="violet" />
            <StatCard label="Pozicione" value={`${totalPositions}`} hint="Në të gjitha ofertat" icon={<IconGrid />} tone="teal" />
            <StatCard label="Mesatarja" value={`€ ${money(avgOffer)}`} hint="Total / ofertë" icon={<IconChart />} tone="amber" />
          </section>

          {/* Controls */}
          <section className="od2Controls">
            <div className="od2ControlsLeft">
              <div className="od2Search">
                <span className="od2SearchIcon">
                  <IconSearch />
                </span>
                <input
                  className="od2SearchInput"
                  value={q}
                  onChange={(e) => setQ(e.target.value)}
                  placeholder="Kërko ofertë…"
                  autoComplete="off"
                />
              </div>

              <div className="od2Sort">
                <span className="od2SortLabel">Renditja</span>
                <Select value={sort} onChange={(e) => setSort(e.target.value)} style={{ height: 40 }}>
                  <option value="updated_desc">Data (më e re)</option>
                  <option value="updated_asc">Data (më e vjetër)</option>
                  <option value="name_asc">Emri (A–Z)</option>
                  <option value="name_desc">Emri (Z–A)</option>
                  <option value="price_desc">Çmimi (më i lartë)</option>
                  <option value="price_asc">Çmimi (më i lirë)</option>
                </Select>
              </div>
            </div>

            <div className="od2ControlsRight">
              <Btn variant="primary" onClick={openCreate} style={{ borderRadius: 12, padding: "10px 14px" }}>
                <span style={{ display: "inline-flex", alignItems: "center", gap: 8 }}>
                  <IconPlus />
                  Krijo ofertë
                </span>
              </Btn>
            </div>
          </section>

          {/* List */}
          <section className="od2List">
            <div className="od2ListHead">
              <div className="od2ListTitle">Lista e ofertave</div>
              <div className="od2ListMeta">
                <Pill tone="soft">{sorted.length} ofertë(a)</Pill>
              </div>
            </div>

            {sorted.length === 0 ? (
              <div className="od2Empty">
                <div className="od2EmptyIcon">
                  <IconSearchBig />
                </div>
                <div className="od2EmptyTitle">Nuk u gjet asnjë ofertë</div>
                <div className="od2EmptySub">Provo një fjalë tjetër ose krijo ofertë të re.</div>
              </div>
            ) : (
              <>
                {/* ✅ TABLE (fixed layout -> kolonat gjithmonë në vijë) */}
                <div className="od2TableWrap">
                  <table className="od2Table">
                    <colgroup>
                      <col className="cName" />
                      <col className="cDesc" />
                      <col className="cSmall" />
                      <col className="cSmall" />
                      <col className="cTotal" />
                      <col className="cDate" />
                      <col className="cAct" />
                    </colgroup>
                    <thead>
                      <tr>
                        <th>Emri</th>
                        <th>Përshkrimi</th>
                        <th className="od2ThSmall">Poz</th>
                        <th className="od2ThSmall">Marzha</th>
                        <th className="od2ThRight">Totali</th>
                        <th className="od2ThDate">Përditësuar</th>
                        <th className="od2ThRight">Veprime</th>
                      </tr>
                    </thead>
                    <tbody>
                      {sorted.map((o) => (
                        <OfferRow
                          key={o.id}
                          offer={o}
                          total={totalsByOfferId.get(o.id) || 0}
                          onOpen={() => onOpenOffer(o.id)}
                        />
                      ))}
                    </tbody>
                  </table>
                </div>

                {/* MOBILE cards */}
                <div className="od2Cards">
                  {sorted.map((o) => (
                    <OfferCard key={o.id} offer={o} total={totalsByOfferId.get(o.id) || 0} onOpen={() => onOpenOffer(o.id)} />
                  ))}
                </div>
              </>
            )}
          </section>
        </div>

        {/* Create modal */}
        <Modal
          open={createOpen}
          title="Krijo ofertë të re"
          onClose={() => setCreateOpen(false)}
          footer={
            <div style={{ display: "flex", justifyContent: "space-between", gap: 10 }}>
              <Btn variant="ghost" onClick={() => setCreateOpen(false)} style={{ borderRadius: 12 }}>
                Anulo
              </Btn>
              <Btn variant="primary" onClick={createOffer} disabled={!name.trim() || !profileId || !ironId} style={{ borderRadius: 12 }}>
                Krijo ofertën
              </Btn>
            </div>
          }
        >
          <div className="od2ModalGrid">
            <Field label="Emri i ofertës">
              <input
                className="od2ModalInput"
                value={name}
                onChange={(e) => setName(e.target.value)}
                placeholder="Emri i Klientit"
                autoFocus
                autoComplete="off"
              />
            </Field>

            <Field label="Përshkrimi (opsional)">
              <textarea
                className="od2ModalInput od2ModalTextarea"
                value={description}
                onChange={(e) => setDescription(e.target.value)}
                placeholder="p.sh. Objekti X / Hyrja A"
                rows={3}
              />
            </Field>

            <div className="od2ModalTwo">
              <Field label="Marzha (%)">
                <input
                  className="od2ModalInput"
                  type="number"
                  value={margin}
                  onChange={(e) => setMargin(Number(e.target.value) || 0)}
                  placeholder="20"
                />
              </Field>

              <Field label="Ngjyra">
                <select className="od2ModalInput" value={color} onChange={(e) => setColor(e.target.value)}>
                  <option value="[BB] Bardhë-Bardhë">[BB] Bardhë-Bardhë</option>
                  <option value="Antracit">Antracit</option>
                  <option value="Golden Oak">Golden Oak</option>
                </select>
              </Field>
            </div>

            <div className="od2ModalTwo">
              <Field label="Profili default">
                <select className="od2ModalInput" value={profileId} onChange={(e) => setProfileId(e.target.value)}>
                  {materials.profiles.map((p) => (
                    <option key={p.id} value={p.id}>
                      {p.name} — € {money(p.pricePerM)}/m
                    </option>
                  ))}
                </select>
              </Field>

              <Field label="Armatura default">
                <select className="od2ModalInput" value={ironId} onChange={(e) => setIronId(e.target.value)}>
                  {materials.irons.map((i) => (
                    <option key={i.id} value={i.id}>
                      {i.name} — € {money(i.pricePerM)}/m
                    </option>
                  ))}
                </select>
              </Field>
            </div>

            <div style={{ fontSize: 12, color: UI.muted }}>
              Çmimet mund t’i menaxhosh në çdo kohë te <b>Materialet</b>.
            </div>
          </div>
        </Modal>
      </main>
    </div>
  );
}

/* ---------------- UI Pieces ---------------- */

function StatCard({ label, value, hint, icon, tone = "blue" }) {
  return (
    <div className={`od2Stat od2Tone-${tone}`}>
      <div className="od2StatTop">
        <div className="od2StatIcon">{icon}</div>
        <div className="od2StatLabel">{label}</div>
      </div>
      <div className="od2StatValue">{value}</div>
      <div className="od2StatHint">{hint}</div>
    </div>
  );
}

function OfferRow({ offer, total, onOpen }) {
  const positions = offer.positions || [];
  const updated = offer.updatedAt ? new Date(offer.updatedAt).toLocaleDateString() : "—";

  return (
    <tr className="od2Row" onClick={onOpen} role="button" tabIndex={0}>
      <td className="tdName" title={offer.name}>
        <div className="od2Name">{offer.name}</div>
        <div className="od2Mini">{positions.length} pozicione</div>
      </td>

      <td className="tdDesc" title={offer.description || ""}>
        <div className="od2Desc">{offer.description || "—"}</div>
      </td>

      <td className="od2TdSmall">{positions.length}</td>
      <td className="od2TdSmall">{offer.margin}%</td>

      <td className="od2TdRight">
        <span className="od2Total">€ {money(total)}</span>
      </td>

      <td className="od2TdDate">{updated}</td>

      <td className="od2TdRight" onClick={(e) => e.stopPropagation()}>
        <Btn variant="soft" onClick={onOpen} style={{ borderRadius: 10, padding: "9px 12px" }}>
          <span style={{ display: "inline-flex", alignItems: "center", gap: 8 }}>
            Hape <IconChevronRight />
          </span>
        </Btn>
      </td>
    </tr>
  );
}

function OfferCard({ offer, total, onOpen }) {
  const positions = offer.positions || [];
  return (
    <div className="od2Card" onClick={onOpen} role="button" tabIndex={0}>
      <div className="od2CardTop">
        <div className="od2CardLeft">
          <div className="od2CardName">{offer.name}</div>
          <div className="od2CardDesc">{offer.description || "—"}</div>
          <div className="od2CardSmall">Poz: {positions.length} • Marzha: {offer.margin}%</div>
        </div>
        <div className="od2CardRight">
          <div className="od2CardTotal">€ {money(total)}</div>
        </div>
      </div>

      <div className="od2CardBottom">
        <div className="od2CardSmall">Përditësuar: {offer.updatedAt ? new Date(offer.updatedAt).toLocaleDateString() : "—"}</div>
        <div onClick={(e) => e.stopPropagation()}>
          <Btn variant="soft" onClick={onOpen} style={{ borderRadius: 10, padding: "9px 12px" }}>
            <span style={{ display: "inline-flex", alignItems: "center", gap: 8 }}>
              Hape <IconChevronRight />
            </span>
          </Btn>
        </div>
      </div>
    </div>
  );
}

/* ---------------- Styles ---------------- */

function styles(UI) {
  const primary = UI.primary || "#2563eb";
  const border = UI.border || "rgba(15,23,42,0.10)";
  const text = UI.text || "#0f172a";
  const muted = UI.muted || "rgba(15,23,42,0.60)";

  return `
  .od2Root{
    min-height: 100vh;
    background: #f6f7fb;
    color: ${text};
    font-family: Inter, ui-sans-serif, system-ui, -apple-system, Segoe UI, Roboto, Helvetica, Arial;
  }

  .od2Nav{
    position: sticky;
    top: 0;
    z-index: 30;
    background: rgba(255,255,255,0.92);
    backdrop-filter: blur(10px);
    border-bottom: 1px solid ${border};
  }
  .od2NavInner{
    width: 100%;
    padding: 12px 18px;
    display:flex;
    align-items:center;
    justify-content:space-between;
    gap: 12px;
  }
  .od2NavLeft{
    display:flex;
    align-items:center;
    gap: 14px;
    min-width:0;
  }

 
  .od2LogoImg{ width: 44px; height: 44px; object-fit: contain; }
  .od2LogoFallback{
    width: 44px;
    height: 44px;
    display:grid;
    place-items:center;
    font-size: 12px;
    letter-spacing: .3px;
    color: ${text};
    background: rgba(37,99,235,0.10);
    border-radius: 14px;
  }

  .od2Links{
    display:flex;
    align-items:center;
    gap: 8px;
    min-width:0;
  }
  .od2Link{
    display:flex;
    align-items:center;
    gap: 10px;
    padding: 10px 12px;
    border-radius: 14px;
    border: 1px solid transparent;
    background: transparent;
    color: ${text};
    cursor:pointer;
    font-size: 13px;
    font-weight: 500;
    transition: background .12s ease, border-color .12s ease, transform .12s ease;
    text-align:left;
  }
  .od2Link:hover{
    background: rgba(37,99,235,0.07);
    border-color: rgba(37,99,235,0.14);
    transform: translateY(-1px);
  }
  .od2Active{
    background: rgba(37,99,235,0.10);
    border-color: rgba(37,99,235,0.18);
  }
  .od2LinkIcon{
    width: 34px;
    height: 34px;
    border-radius: 12px;
    display:grid;
    place-items:center;
    background: rgba(15,23,42,0.04);
    border: 1px solid ${border};
    color: ${primary};
    flex: 0 0 auto;
  }

  .od2NavRight{ display:flex; align-items:center; gap: 10px; }
  .od2UserBtn{
    width: 44px;
    height: 44px;
    border-radius: 14px;
    border: 1px solid ${border};
    background: #fff;
    box-shadow: 0 8px 18px rgba(15,23,42,0.06);
    cursor:pointer;
    display:grid;
    place-items:center;
    transition: transform .12s ease, background .12s ease;
  }
  .od2UserBtn:hover{
    transform: translateY(-1px);
    background: rgba(15,23,42,0.02);
  }
  .od2UserIcon{ color: rgba(15,23,42,0.75); display:grid; place-items:center; }

  .od2Main{ padding: 18px; }
  .od2Full{
    width: 100%;
    display:grid;
    gap: 14px;
  }

  /* Stats */
  .od2Stats{
    display:grid;
    grid-template-columns: repeat(4, minmax(0,1fr));
    gap: 12px;
  }
  .od2Stat{
    background: #fff;
    border: 1px solid ${border};
    border-radius: 18px;
    box-shadow: 0 10px 22px rgba(15,23,42,0.06);
    padding: 14px;
    min-width:0;
  }
  .od2StatTop{ display:flex; align-items:center; gap: 10px; }
  .od2StatIcon{
    width: 38px;
    height: 38px;
    border-radius: 14px;
    display:grid;
    place-items:center;
    border: 1px solid ${border};
    background: rgba(15,23,42,0.03);
  }
  .od2StatLabel{ font-size: 12px; color: ${muted}; font-weight: 500; }
  .od2StatValue{ margin-top: 10px; font-size: 18px; font-weight: 600; letter-spacing: .1px; }
  .od2StatHint{ margin-top: 8px; font-size: 12px; color: ${muted}; white-space: nowrap; overflow:hidden; text-overflow: ellipsis; }

  .od2Tone-blue .od2StatIcon{ color: #2563eb; background: rgba(37,99,235,0.10); border-color: rgba(37,99,235,0.16); }
  .od2Tone-violet .od2StatIcon{ color: #7c3aed; background: rgba(124,58,237,0.10); border-color: rgba(124,58,237,0.16); }
  .od2Tone-teal .od2StatIcon{ color: #0f766e; background: rgba(15,118,110,0.10); border-color: rgba(15,118,110,0.16); }
  .od2Tone-amber .od2StatIcon{ color: #b45309; background: rgba(180,83,9,0.10); border-color: rgba(180,83,9,0.16); }

  /* Controls */
  .od2Controls{
    display:flex;
    align-items:center;
    justify-content:space-between;
    gap: 12px;
  }
  .od2ControlsLeft{
    display:flex;
    align-items:center;
    gap: 12px;
    min-width:0;
  }
  .od2Search{
    display:flex;
    align-items:center;
    gap: 10px;
    height: 42px;
    border-radius: 16px;
    border: 1px solid ${border};
    background: #fff;
    padding: 0 12px;
    box-shadow: 0 10px 20px rgba(15,23,42,0.05);
    min-width: 360px;
    max-width: 46vw;
  }
  .od2SearchIcon{ color: rgba(15,23,42,0.50); display:grid; place-items:center; }
  .od2SearchInput{
    width: 100%;
    height: 40px;
    border: none;
    outline: none;
    font-size: 13px;
    background: transparent;
  }

  .od2Sort{
    display:flex;
    align-items:center;
    gap: 10px;
    height: 42px;
    border-radius: 16px;
    border: 1px solid ${border};
    background: #fff;
    padding: 0 12px;
    box-shadow: 0 10px 20px rgba(15,23,42,0.05);
  }
  .od2SortLabel{ font-size: 12px; color: ${muted}; font-weight: 500; white-space: nowrap; }

  /* List */
  .od2List{
    background: #fff;
    border: 1px solid ${border};
    border-radius: 18px;
    box-shadow: 0 10px 22px rgba(15,23,42,0.06);
    overflow:hidden;
  }
  .od2ListHead{
    padding: 14px 14px 10px 14px;
    display:flex;
    align-items:center;
    justify-content:space-between;
    gap: 12px;
    border-bottom: 1px solid ${border};
    background: linear-gradient(180deg, rgba(15,23,42,0.02), rgba(15,23,42,0.00));
  }
  .od2ListTitle{ font-size: 13px; font-weight: 600; }
  .od2ListMeta{ display:flex; align-items:center; gap: 10px; }

  .od2TableWrap{ width:100%; overflow:auto; }
  .od2Table{
    width:100%;
    border-collapse: separate;
    border-spacing: 0;
    table-layout: fixed; /* ✅ FIX */
  }

  /* col widths */
  .cName{ width: 260px; }
  .cDesc{ width: 320px; }
  .cSmall{ width: 90px; }
  .cTotal{ width: 150px; }
  .cDate{ width: 150px; }
  .cAct{ width: 160px; }

  .od2Table thead th{
    font-size: 12px;
    color: ${muted};
    font-weight: 700;
    text-align:left;
    padding: 10px 14px;
    white-space: nowrap;
    border-bottom: 1px solid ${border};
    background: #fff;
    position: sticky;
    top: 0;
    z-index: 1;
  }
  .od2Row td{
    padding: 10px 14px;
    border-bottom: 1px solid rgba(15,23,42,0.06);
    font-size: 13px;
    vertical-align: middle;
    overflow: hidden; /* ✅ keep inside cells */
  }
  .od2Row{ cursor:pointer; transition: background .12s ease; }
  .od2Row:hover{ background: rgba(37,99,235,0.05); }

  .od2Name{
    font-weight: 700;
    white-space: nowrap;
    overflow:hidden;
    text-overflow: ellipsis;
  }
  .od2Mini{
    margin-top: 4px;
    font-size: 12px;
    color: ${muted};
    white-space: nowrap;
    overflow:hidden;
    text-overflow: ellipsis;
  }
  .od2Desc{
    white-space: nowrap;
    overflow:hidden;
    text-overflow: ellipsis;
    color: rgba(15,23,42,0.80);
  }
  .od2Total{ font-weight: 750; }
  .od2ThSmall, .od2TdSmall{ width: 90px; }
  .od2ThRight, .od2TdRight{ white-space: nowrap; }
  .od2ThDate, .od2TdDate{ width: 150px; white-space: nowrap; color: ${muted}; font-size: 12px; }

  .od2Cards{ display:none; padding: 12px; gap: 12px; }
  .od2Card{
    border: 1px solid ${border};
    border-radius: 18px;
    padding: 12px;
    background: #fff;
    box-shadow: 0 10px 22px rgba(15,23,42,0.06);
  }
  .od2CardTop{ display:flex; justify-content:space-between; gap: 12px; align-items:flex-start; }
  .od2CardLeft{ min-width:0; }
  .od2CardName{ font-weight: 750; font-size: 14px; }
  .od2CardDesc{ margin-top: 6px; color: ${muted}; font-size: 12px; white-space: nowrap; overflow:hidden; text-overflow: ellipsis; max-width: 520px; }
  .od2CardRight{ display:flex; flex-direction:column; align-items:flex-end; gap: 8px; }
  .od2CardTotal{ font-weight: 800; white-space: nowrap; }
  .od2CardSmall{ margin-top: 8px; color: ${muted}; font-size: 12px; }
  .od2CardBottom{ margin-top: 12px; display:flex; justify-content:space-between; align-items:center; gap: 10px; }

  /* Empty */
  .od2Empty{ padding: 28px 16px; text-align:center; color: ${muted}; }
  .od2EmptyIcon{ display:grid; place-items:center; margin-bottom: 10px; opacity: .9; }
  .od2EmptyTitle{ font-size: 14px; font-weight: 750; color: ${text}; }
  .od2EmptySub{ margin-top: 6px; font-size: 12px; }

  /* Modal inputs */
  .od2ModalGrid{ display:grid; gap: 12px; }
  .od2ModalTwo{ display:grid; grid-template-columns: 1fr 1fr; gap: 12px; }
  .od2ModalInput{
    width: 100%;
    height: 42px;
    border: 1px solid ${border};
    border-radius: 12px;
    padding: 0 12px;
    font-size: 13px;
    outline: none;
    box-sizing: border-box;
    background: #fff;
  }
  .od2ModalInput:focus{
    border-color: rgba(37,99,235,0.30);
    box-shadow: 0 0 0 4px rgba(37,99,235,0.10);
  }
  .od2ModalTextarea{
    height: auto;
    padding: 10px 12px;
    resize: vertical;
    line-height: 1.4;
  }

  /* Responsive */
  @media (max-width: 1060px){
    .od2Stats{ grid-template-columns: repeat(2, minmax(0,1fr)); }
    .od2Search{ min-width: 260px; }
    .cDesc{ width: 260px; }
  }
  @media (max-width: 820px){
    .od2Controls{ flex-direction: column; align-items: stretch; }
    .od2ControlsLeft{ flex-direction: column; align-items: stretch; }
    .od2Search{ max-width: 100%; min-width: 100%; }
    .od2Sort{ width: 100%; }
    .od2ControlsRight{ display:flex; justify-content:flex-end; }
    .od2ModalTwo{ grid-template-columns: 1fr; }
  }
  @media (max-width: 900px){
    .od2TableWrap{ display:none; }
    .od2Cards{ display:grid; }
  }
  `;
}

/* ---------------- Icons ---------------- */
function IconHome() {
  return (
    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" aria-hidden="true">
      <path d="M4 10.5 12 4l8 6.5V20a1.5 1.5 0 0 1-1.5 1.5H5.5A1.5 1.5 0 0 1 4 20v-9.5Z" stroke="currentColor" strokeWidth="2" strokeLinejoin="round" />
      <path d="M9 21v-7a1 1 0 0 1 1-1h4a1 1 0 0 1 1 1v7" stroke="currentColor" strokeWidth="2" strokeLinejoin="round" />
    </svg>
  );
}
function IconSliders() {
  return (
    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" aria-hidden="true">
      <path d="M4 6h10M18 6h2" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
      <path d="M14 6a2 2 0 1 0 4 0a2 2 0 0 0-4 0Z" stroke="currentColor" strokeWidth="2" />
      <path d="M4 12h2M10 12h10" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
      <path d="M6 12a2 2 0 1 0 4 0a2 2 0 0 0-4 0Z" stroke="currentColor" strokeWidth="2" />
      <path d="M4 18h10M18 18h2" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
      <path d="M14 18a2 2 0 1 0 4 0a2 2 0 0 0-4 0Z" stroke="currentColor" strokeWidth="2" />
    </svg>
  );
}
function IconUser() {
  return (
    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" aria-hidden="true">
      <path d="M20 21a8 8 0 0 0-16 0" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
      <path d="M12 12a4 4 0 1 0 0-8a4 4 0 0 0 0 8Z" stroke="currentColor" strokeWidth="2" />
    </svg>
  );
}
function IconSearch() {
  return (
    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" aria-hidden="true">
      <path d="M10.5 18a7.5 7.5 0 1 1 0-15a7.5 7.5 0 0 1 0 15Z" stroke="currentColor" strokeWidth="2" />
      <path d="M16.5 16.5L21 21" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
    </svg>
  );
}
function IconSearchBig() {
  return (
    <svg width="26" height="26" viewBox="0 0 24 24" fill="none" aria-hidden="true">
      <path d="M10.5 18a7.5 7.5 0 1 1 0-15a7.5 7.5 0 0 1 0 15Z" stroke="currentColor" strokeWidth="2" />
      <path d="M16.5 16.5L21 21" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
    </svg>
  );
}
function IconPlus() {
  return (
    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" aria-hidden="true">
      <path d="M12 5v14M5 12h14" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
    </svg>
  );
}
function IconChevronRight() {
  return (
    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" aria-hidden="true">
      <path d="M9 18l6-6-6-6" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
    </svg>
  );
}
function IconEuro() {
  return (
    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" aria-hidden="true">
      <path d="M18 7a6 6 0 1 0 0 10" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
      <path d="M5 10h10M5 14h10" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
    </svg>
  );
}
function IconFile() {
  return (
    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" aria-hidden="true">
      <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-8-6Z" stroke="currentColor" strokeWidth="2" />
      <path d="M14 2v6h8" stroke="currentColor" strokeWidth="2" />
    </svg>
  );
}
function IconGrid() {
  return (
    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" aria-hidden="true">
      <path d="M3 3h8v8H3V3Zm10 0h8v8h-8V3ZM3 13h8v8H3v-8Zm10 0h8v8h-8v-8Z" stroke="currentColor" strokeWidth="2" />
    </svg>
  );
}
function IconChart() {
  return (
    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" aria-hidden="true">
      <path d="M4 19V5" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
      <path d="M4 19h16" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
      <path d="M8 15l3-3 3 2 5-6" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
    </svg>
  );
}
